/**
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.activemq.broker;

import org.apache.activemq.store.PersistenceAdapter;
import org.apache.activemq.util.ServiceStopper;
import org.apache.activemq.util.ServiceSupport;
import org.apache.activemq.util.ThreadPoolUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.IOException;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.ScheduledThreadPoolExecutor;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.TimeUnit;

/**
 * Helper class for working with services that requires locking
 */
public abstract class LockableServiceSupport extends ServiceSupport implements Lockable, BrokerServiceAware {

    private static final Logger LOG = LoggerFactory.getLogger(LockableServiceSupport.class);
    boolean useLock = true;
    Locker locker;
    long lockKeepAlivePeriod = 0;
    private ScheduledFuture<?> keepAliveTicket;
    private ScheduledThreadPoolExecutor clockDaemon;
    private BrokerService brokerService;

    /**
     * Initialize resources before locking
     *
     * @throws Exception
     */
    abstract public void init() throws Exception;

    @Override
    public void setUseLock(boolean useLock) {
        this.useLock = useLock;
    }

    @Override
    public void setLocker(Locker locker) throws IOException {
        this.locker = locker;
        if (this instanceof PersistenceAdapter) {
            this.locker.configure((PersistenceAdapter)this);
        }
    }

    public Locker getLocker() throws IOException {
        if (this.locker == null) {
            this.locker = createDefaultLocker();
        }
        return this.locker;
    }

    @Override
    public void setLockKeepAlivePeriod(long lockKeepAlivePeriod) {
        this.lockKeepAlivePeriod = lockKeepAlivePeriod;
    }

    @Override
    public void preStart() throws Exception {
        init();
        if (useLock) {
            if (getLocker() == null) {
                LOG.warn("No locker configured");
            } else {
                getLocker().start();
                if (lockKeepAlivePeriod > 0) {
                    keepAliveTicket = getScheduledThreadPoolExecutor().scheduleAtFixedRate(new Runnable() {
                        public void run() {
                            keepLockAlive();
                        }
                    }, lockKeepAlivePeriod, lockKeepAlivePeriod, TimeUnit.MILLISECONDS);
                }
                if (brokerService != null) {
                    brokerService.getBroker().nowMasterBroker();
                }
            }
        }
    }

    @Override
    public void postStop(ServiceStopper stopper) throws Exception {
        if (useLock) {
            if (keepAliveTicket != null) {
                keepAliveTicket.cancel(false);
                keepAliveTicket = null;
            }
            if (locker != null) {
                getLocker().stop();
            }
            ThreadPoolUtils.shutdown(clockDaemon);
        }
    }

    protected void keepLockAlive() {
        boolean stop = false;
        try {
            Locker locker = getLocker();
            if (locker != null) {
                if (!locker.keepAlive()) {
                    stop = true;
                }
            }
        } catch (IOException e) {
            LOG.warn("locker keepalive resulted in: " + e, e);
        }
        if (stop) {
            stopBroker();
        }
    }

    protected void stopBroker() {
        // we can no longer keep the lock so lets fail
        LOG.info(brokerService.getBrokerName() + ", no longer able to keep the exclusive lock so giving up being a master");
        try {
            brokerService.stop();
        } catch (Exception e) {
            LOG.warn("Failure occurred while stopping broker");
        }
    }

    public ScheduledThreadPoolExecutor getScheduledThreadPoolExecutor() {
        if (clockDaemon == null) {
            clockDaemon = new ScheduledThreadPoolExecutor(5, new ThreadFactory() {
                public Thread newThread(Runnable runnable) {
                    Thread thread = new Thread(runnable, "ActiveMQ Lock KeepAlive Timer");
                    thread.setDaemon(true);
                    return thread;
                }
            });
        }
        return clockDaemon;
    }

    @Override
    public void setBrokerService(BrokerService brokerService) {
        this.brokerService = brokerService;
    }
}
