/*
 * Copyright Alibaba Group Holding Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.alibaba.hbase.client;

import static com.alibaba.hbase.client.Constants.ALIHBASE_CLIENT_SCANNER_CACHING;
import static com.alibaba.hbase.client.Constants.ALIHBASE_CLIENT_SCANNER_CACHING_DEFAULT;

import java.io.IOException;
import java.nio.ByteBuffer;
import java.util.ArrayDeque;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Queue;

import com.alibaba.hbase.thrift2.generated.TAppend;
import com.alibaba.hbase.thrift2.generated.TColumnIncrement;
import com.alibaba.hbase.thrift2.generated.TDelete;
import com.alibaba.hbase.thrift2.generated.TGet;
import com.alibaba.hbase.thrift2.generated.THBaseService;
import com.alibaba.hbase.thrift2.generated.TIncrement;
import com.alibaba.hbase.thrift2.generated.TPut;
import com.alibaba.hbase.thrift2.generated.TResult;
import com.alibaba.hbase.thrift2.generated.TRowMutations;
import com.alibaba.hbase.thrift2.generated.TScan;
import com.alibaba.hbase.thrift2.generated.TTableDescriptor;
import com.google.protobuf.Descriptors;
import com.google.protobuf.Message;
import com.google.protobuf.Service;
import com.google.protobuf.ServiceException;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.hbase.HConstants;
import org.apache.hadoop.hbase.HTableDescriptor;
import org.apache.hadoop.hbase.TableName;
import org.apache.hadoop.hbase.client.Append;
import org.apache.hadoop.hbase.client.Delete;
import org.apache.hadoop.hbase.client.Durability;
import org.apache.hadoop.hbase.client.Get;
import org.apache.hadoop.hbase.client.Increment;
import org.apache.hadoop.hbase.client.Put;
import org.apache.hadoop.hbase.client.Result;
import org.apache.hadoop.hbase.client.ResultScanner;
import org.apache.hadoop.hbase.client.Row;
import org.apache.hadoop.hbase.client.RowMutations;
import org.apache.hadoop.hbase.client.Scan;
import org.apache.hadoop.hbase.client.Table;
import org.apache.hadoop.hbase.client.coprocessor.Batch;
import org.apache.hadoop.hbase.client.metrics.ScanMetrics;
import org.apache.hadoop.hbase.filter.CompareFilter;
import org.apache.hadoop.hbase.io.TimeRange;
import org.apache.hadoop.hbase.ipc.CoprocessorRpcChannel;
import org.apache.hadoop.hbase.util.Bytes;
import org.apache.thrift.TException;
import org.apache.thrift.transport.TTransport;

public class ThriftTable implements Table {

  private TableName tableName;
  private Configuration conf;
  private TTransport tTransport;
  private THBaseService.Client client;
  private ByteBuffer tableNameInBytes;
  private int operationTimeout;

  private final int scannerCaching;

  public ThriftTable(TableName tableName, THBaseService.Client client, TTransport tTransport,
      Configuration conf) {
    this.tableName = tableName;
    this.tableNameInBytes = ByteBuffer.wrap(tableName.toBytes());
    this.conf = conf;
    this.tTransport = tTransport;
    this.client = client;
    this.scannerCaching = conf.getInt(ALIHBASE_CLIENT_SCANNER_CACHING,
        ALIHBASE_CLIENT_SCANNER_CACHING_DEFAULT);
    this.operationTimeout = conf.getInt(HConstants.HBASE_CLIENT_OPERATION_TIMEOUT,
        HConstants.DEFAULT_HBASE_CLIENT_OPERATION_TIMEOUT);


  }

  @Override
  public TableName getName() {
    return tableName;
  }

  @Override
  public Configuration getConfiguration() {
    return conf;
  }

  @Override
  public HTableDescriptor getTableDescriptor() throws IOException {
    try {
      TTableDescriptor tableDescriptor = client
          .getTableDescriptor(ThriftUtilities.tableNameFromHBase(tableName));
      return ThriftUtilities.tableDescriptorFromThrift(tableDescriptor);
    } catch (TException e) {
      throw new IOException(e);
    }
  }

  @Override
  public boolean exists(Get get) throws IOException {
    TGet tGet = ThriftUtilities.getFromHBase(get);
    try {
      return client.exists(tableNameInBytes, tGet);
    }  catch (TException e) {
      throw new IOException(e);
    }
  }

  @Override
  public boolean[] existsAll(List<Get> gets) throws IOException {
    List<TGet> tGets = new ArrayList<>();
    for (Get get: gets) {
      tGets.add(ThriftUtilities.getFromHBase(get));
    }
    try {
      List<Boolean> results = client.existsAll(tableNameInBytes, tGets);
      boolean[] booleans = new boolean[results.size()];
      for (int i = 0; i < results.size(); i++) {
        booleans[i] = results.get(i);
      }
      return booleans;
    }  catch (TException e) {
      throw new IOException(e);
    }
  }

  @Override
  public void batch(List<? extends Row> actions, Object[] results)
      throws IOException {
    throw new IOException("Batch not supported in ThriftTable, use put(List<Put> puts), "
        + "get(List<Get> gets) or delete(List<Delete> deletes) respectively");


  }

  @Override
  public <R> void batchCallback(List<? extends Row> actions, Object[] results,
      Batch.Callback<R> callback) throws IOException {
    throw new IOException("BatchCallback not supported in ThriftTable, use put(List<Put> puts), "
        + "get(List<Get> gets) or delete(List<Delete> deletes) respectively");
  }

  @Override
  public Result get(Get get) throws IOException {
    TGet tGet = ThriftUtilities.getFromHBase(get);
    try {
      TResult tResult = client.get(tableNameInBytes, tGet);
      return ThriftUtilities.resultFromThrift(tResult);
    }  catch (TException e) {
      throw new IOException(e);
    }
  }

  @Override
  public Result[] get(List<Get> gets) throws IOException {
    List<TGet> tGets = ThriftUtilities.getsFromHBase(gets);
    try {
      List<TResult> results = client.getMultiple(tableNameInBytes, tGets);
      return ThriftUtilities.resultsFromThrift(results);
    }  catch (TException e) {
      throw new IOException(e);
    }
  }

  /**
   * A scanner to perform scan from thrift server
   * getScannerResults is used in this scanner
   */
  private class Scanner implements ResultScanner {
    protected TScan scan;
    protected Result lastResult = null;
    protected final Queue<Result> cache = new ArrayDeque<>();;


    public Scanner(Scan scan) throws IOException {
      if (scan.getCaching() <= 0) {
        scan.setCaching(scannerCaching);
      } else if (scan.getCaching() == 1 && scan.isReversed()){
        // for reverse scan, we need to pass the last row to the next scanner
        // we need caching number bigger than 1
        scan.setCaching(scan.getCaching() + 1);
      }
      this.scan = ThriftUtilities.scanFromHBase(scan);
    }

    @Override
    public Result[] next(int nbRows) throws IOException {
      // Collect values to be returned here
      ArrayList<Result> resultSets = new ArrayList<Result>(nbRows);
      for(int i = 0; i < nbRows; i++) {
        Result next = next();
        if (next != null) {
          resultSets.add(next);
        } else {
          break;
        }
      }
      return resultSets.toArray(new Result[resultSets.size()]);
    }

    @Override
    public Iterator<Result> iterator() {
      return new Iterator<Result>() {
        // The next RowResult, possibly pre-read
        Result next = null;

        // return true if there is another item pending, false if there isn't.
        // this method is where the actual advancing takes place, but you need
        // to call next() to consume it. hasNext() will only advance if there
        // isn't a pending next().
        @Override
        public boolean hasNext() {
          if (next == null) {
            try {
              next = Scanner.this.next();
              return next != null;
            } catch (IOException e) {
              throw new RuntimeException(e);
            }
          }
          return true;
        }

        // get the pending next item and advance the iterator. returns null if
        // there is no next item.
        @Override
        public Result next() {
          // since hasNext() does the real advancing, we call this to determine
          // if there is a next before proceeding.
          if (!hasNext()) {
            return null;
          }

          // if we get to here, then hasNext() has given us an item to return.
          // we want to return the item and then null out the next pointer, so
          // we use a temporary variable.
          Result temp = next;
          next = null;
          return temp;
        }

        @Override
        public void remove() {
          throw new UnsupportedOperationException();
        }
      };
    }

    @Override
    public Result next() throws IOException {
      if (cache.size() == 0) {
        setupNextScanner();
        try {
          List<TResult> tResults = client
              .getScannerResults(tableNameInBytes, scan, scan.getCaching());
          Result[] results = ThriftUtilities.resultsFromThrift(tResults);
          boolean firstKey = true;
          for (Result result : results) {
            // If it is a reverse scan, we use the last result's key as the startkey, since there is
            // no way to construct a closet rowkey smaller than the last result
            // So when the results return, we must rule out the first result, since it has already
            // returned to user.
            if (firstKey) {
              firstKey = false;
              if (scan.isReversed() && lastResult != null) {
                if (Bytes.equals(lastResult.getRow(), result.getRow())) {
                  continue;
                }
              }
            }
            cache.add(result);
            lastResult = result;
          }
        } catch (TException e) {
          throw new IOException(e);
        }
      }

      if (cache.size() > 0) {
        return cache.poll();
      } else {
        //scan finished
        return null;
      }
    }

    @Override
    public void close() {
    }


    public boolean renewLease() {
      throw new RuntimeException("renewLease() not supported");
    }

    public ScanMetrics getScanMetrics() {
      throw new RuntimeException("getScanMetrics() not supported");
    }

    private void setupNextScanner() {
      //if lastResult is null null, it means it is not the fist scan
      if (lastResult!= null) {
        byte[] lastRow = lastResult.getRow();
        if (scan.isReversed()) {
          //for reverse scan, we can't find the closet row before this row
          scan.setStartRow(lastRow);
        } else {
          scan.setStartRow(createClosestRowAfter(lastRow));
        }
      }
    }


    /**
     * Create the closest row after the specified row
     */
    protected byte[] createClosestRowAfter(byte[] row) {
      if (row == null) {
        throw new RuntimeException("The passed row is null");
      }
      return Arrays.copyOf(row, row.length + 1);
    }
  }

  @Override
  public ResultScanner getScanner(Scan scan) throws IOException {
    return new Scanner(scan);
  }

  @Override
  public ResultScanner getScanner(byte[] family) throws IOException {
    Scan scan = new Scan();
    scan.addFamily(family);
    return getScanner(scan);
  }

  @Override
  public ResultScanner getScanner(byte[] family, byte[] qualifier) throws IOException {
    Scan scan = new Scan();
    scan.addColumn(family, qualifier);
    return getScanner(scan);
  }

  @Override
  public void put(Put put) throws IOException {
    TPut tPut = ThriftUtilities.putFromHBase(put);
    try {
      client.put(tableNameInBytes, tPut);
    }  catch (TException e) {
      throw new IOException(e);
    }
  }

  @Override
  public void put(List<Put> puts) throws IOException {
    List<TPut> tPuts = ThriftUtilities.putsFromHBase(puts);
    try {
      client.putMultiple(tableNameInBytes, tPuts);
    }  catch (TException e) {
      throw new IOException(e);
    }
  }

  @Override
  public void delete(Delete delete) throws IOException {
    TDelete tDelete = ThriftUtilities.deleteFromHBase(delete);
    try {
      client.deleteSingle(tableNameInBytes, tDelete);
    }  catch (TException e) {
      throw new IOException(e);
    }
  }

  @Override
  public void delete(List<Delete> deletes) throws IOException {
    List<TDelete> tDeletes = ThriftUtilities.deletesFromHBase(deletes);
    try {
      client.deleteMultiple(tableNameInBytes, tDeletes);
    }  catch (TException e) {
      throw new IOException(e);
    }
  }

  @Override
  public boolean checkAndMutate(byte[] row, byte[] family, byte[] qualifier,  CompareFilter.CompareOp op,
      byte[] value, RowMutations mutation) throws IOException {
    try {
      ByteBuffer valueBuffer = value == null? null : ByteBuffer.wrap(value);
      return client.checkAndMutate(tableNameInBytes, ByteBuffer.wrap(row), ByteBuffer.wrap(family),
          ByteBuffer.wrap(qualifier), ThriftUtilities.compareOpFromHBase(op), valueBuffer,
          ThriftUtilities.rowMutationsFromHBase(mutation));
    } catch (TException e) {
      throw new IOException(e);
    }
  }

  @Override
  public boolean checkAndPut(final byte [] row,
      final byte [] family, final byte [] qualifier, final byte [] value,
      final Put put)
      throws IOException {
    try {
      ByteBuffer valueBuffer = value == null ? null : ByteBuffer.wrap(value);
      return client.checkAndPut(tableNameInBytes, ByteBuffer.wrap(row), ByteBuffer.wrap(family),
          ByteBuffer.wrap(qualifier), valueBuffer, ThriftUtilities.putFromHBase(put));
    } catch (TException e) {
      throw new IOException(e);
    }
  }

  @Override
  public boolean checkAndDelete(final byte [] row,
      final byte [] family, final byte [] qualifier, final byte [] value,
      final Delete delete) throws IOException {
    try {
      ByteBuffer valueBuffer = value == null ? null : ByteBuffer.wrap(value);
      return client.checkAndDelete(tableNameInBytes, ByteBuffer.wrap(row), ByteBuffer.wrap(family),
          ByteBuffer.wrap(qualifier), valueBuffer, ThriftUtilities.deleteFromHBase(delete));
    } catch (TException e) {
      throw new IOException(e);
    }
  }

  @Override
  public long incrementColumnValue(final byte [] row, final byte [] family,
      final byte [] qualifier, final long amount)
      throws IOException {
    return incrementColumnValue(row, family, qualifier, amount, Durability.SYNC_WAL);
  }

  @Override
  public long incrementColumnValue(final byte [] row, final byte [] family,
      final byte [] qualifier, final long amount, final Durability durability) throws IOException {
    try {
      TIncrement increment = new TIncrement();
      increment.setRow(row);
      TColumnIncrement columnValue = new TColumnIncrement();
      columnValue.setFamily(family).setQualifier(qualifier);
      columnValue.setAmount(amount);
      increment.addToColumns(columnValue);
      increment.setDurability(ThriftUtilities.durabilityFromHBase(durability));
      TResult tResult = client.increment(tableNameInBytes, increment);
      Result result = ThriftUtilities.resultFromThrift(tResult);
      return Long.valueOf(Bytes.toLong(result.getValue(family, qualifier)));
    } catch (TException e) {
      throw new IOException(e);
    }
  }

  @Override
  public void mutateRow(RowMutations rm) throws IOException {
    TRowMutations tRowMutations = ThriftUtilities.rowMutationsFromHBase(rm);
    try {
      client.mutateRow(tableNameInBytes, tRowMutations);
    }  catch (TException e) {
      throw new IOException(e);
    }
  }

  @Override
  public Result append(Append append) throws IOException {
    TAppend tAppend = ThriftUtilities.appendFromHBase(append);
    try {
      TResult tResult = client.append(tableNameInBytes, tAppend);
      return ThriftUtilities.resultFromThrift(tResult);
    }  catch (TException e) {
      throw new IOException(e);
    }
  }

  @Override
  public Result increment(Increment increment) throws IOException {
    TIncrement tIncrement = ThriftUtilities.incrementFromHBase(increment);
    try {
      TResult tResult = client.increment(tableNameInBytes, tIncrement);
      return ThriftUtilities.resultFromThrift(tResult);
    }  catch (TException e) {
      throw new IOException(e);
    }
  }

  @Override
  public void close() throws IOException {
    tTransport.close();
  }

  @Override
  public boolean checkAndPut(byte[] bytes, byte[] bytes1, byte[] bytes2,
      CompareFilter.CompareOp compareOp, byte[] bytes3, Put put) throws IOException {
    throw new UnsupportedOperationException("checkAndPut with compareOp not supported in ThriftTable");
  }

  @Override
  public boolean checkAndDelete(byte[] bytes, byte[] bytes1, byte[] bytes2,
      CompareFilter.CompareOp compareOp, byte[] bytes3, Delete delete) throws IOException {
    throw new UnsupportedOperationException("checkAndDelete with compareOp not supported in ThriftTable");
  }

  @Override
  public CoprocessorRpcChannel coprocessorService(byte[] row) {
    throw new UnsupportedOperationException("coprocessorService not supported in ThriftTable");
  }

  @Override
  public Object[] batch(List<? extends Row> list) throws IOException, InterruptedException {
    throw new UnsupportedOperationException("batch not supported in ThriftTable, use put(list<Put>) instead");
  }

  @Override
  public <R> Object[] batchCallback(List<? extends Row> list, Batch.Callback<R> callback)
      throws IOException, InterruptedException {
    throw new UnsupportedOperationException("batchCallback not supported in ThriftTable");
  }



  @Override
  public <T extends Service, R> Map<byte[], R> coprocessorService(Class<T> aClass, byte[] bytes,
      byte[] bytes1, Batch.Call<T, R> call) throws ServiceException, Throwable {
    throw new UnsupportedOperationException("coprocessorService not supported in ThriftTable");
  }

  @Override
  public <T extends Service, R> void coprocessorService(Class<T> aClass, byte[] bytes,
      byte[] bytes1, Batch.Call<T, R> call, Batch.Callback<R> callback)
      throws ServiceException, Throwable {
    throw new UnsupportedOperationException("coprocessorService not supported in ThriftTable");
  }

  @Override
  public long getWriteBufferSize() {
    throw new UnsupportedOperationException("getWriteBufferSize not supported in ThriftTable");
  }

  @Override
  public void setWriteBufferSize(long l) throws IOException {
    throw new UnsupportedOperationException("setWriteBufferSize not supported in ThriftTable");
  }

  @Override
  public <R extends Message> Map<byte[], R> batchCoprocessorService(
      Descriptors.MethodDescriptor methodDescriptor, Message message, byte[] bytes, byte[] bytes1,
      R r) throws ServiceException, Throwable {
    throw new UnsupportedOperationException("batchCoprocessorService not supported in ThriftTable");
  }

  @Override
  public <R extends Message> void batchCoprocessorService(
      Descriptors.MethodDescriptor methodDescriptor, Message message, byte[] bytes, byte[] bytes1,
      R r, Batch.Callback<R> callback) throws ServiceException, Throwable {
    throw new UnsupportedOperationException("batchCoprocessorService not supported in ThriftTable");
  }

  public void setOperationTimeout(int i) {
    throw new UnsupportedOperationException("setOperationTimeout not supported in ThriftTable");
  }

  public int getOperationTimeout() {
    throw new UnsupportedOperationException("getOperationTimeout not supported in ThriftTable");
  }

  public int getRpcTimeout() {
    throw new UnsupportedOperationException("getRpcTimeout not supported in ThriftTable");
  }

  public void setRpcTimeout(int i) {
    throw new UnsupportedOperationException("setRpcTimeout not supported in ThriftTable");
  }

  public int getReadRpcTimeout() {
    throw new UnsupportedOperationException("getReadRpcTimeout not supported in ThriftTable");
  }

  public void setReadRpcTimeout(int i) {
    throw new UnsupportedOperationException("setReadRpcTimeout not supported in ThriftTable");
  }

  public int getWriteRpcTimeout() {
    throw new UnsupportedOperationException("getWriteRpcTimeout not supported in ThriftTable");
  }

  public void setWriteRpcTimeout(int i) {
    throw new UnsupportedOperationException("setWriteRpcTimeout not supported in ThriftTable");
  }
}
