/*
 * Copyright Alibaba Group Holding Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.alibaba.hbase.client;

import java.io.IOException;
import java.util.ArrayDeque;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Queue;

import com.alibaba.hbase.exception.BatchExceptions;
import com.alibaba.lindorm.client.TableService;
import com.alibaba.lindorm.client.WideColumnService;
import com.alibaba.lindorm.client.core.widecolumnservice.WAppend;
import com.alibaba.lindorm.client.core.widecolumnservice.WDelete;
import com.alibaba.lindorm.client.core.widecolumnservice.WGet;
import com.alibaba.lindorm.client.core.widecolumnservice.WIncrement;
import com.alibaba.lindorm.client.core.widecolumnservice.WPut;
import com.alibaba.lindorm.client.core.widecolumnservice.WResult;
import com.alibaba.lindorm.client.core.widecolumnservice.WRowMutations;
import com.alibaba.lindorm.client.core.widecolumnservice.WScan;
import com.alibaba.lindorm.client.core.widecolumnservice.WScanner;
import com.alibaba.lindorm.client.core.widecolumnservice.filter.WCompareFilter;
import com.alibaba.lindorm.client.schema.LindormTableDescriptor;
import com.google.protobuf.Descriptors;
import com.google.protobuf.Message;
import com.google.protobuf.Service;
import com.google.protobuf.ServiceException;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.hbase.HConstants;
import org.apache.hadoop.hbase.HRegionInfo;
import org.apache.hadoop.hbase.HRegionLocation;
import org.apache.hadoop.hbase.HTableDescriptor;
import org.apache.hadoop.hbase.ServerName;
import org.apache.hadoop.hbase.TableName;
import org.apache.hadoop.hbase.client.Append;
import org.apache.hadoop.hbase.client.Delete;
import org.apache.hadoop.hbase.client.Durability;
import org.apache.hadoop.hbase.client.Get;
import org.apache.hadoop.hbase.client.Increment;
import org.apache.hadoop.hbase.client.Mutation;
import org.apache.hadoop.hbase.client.Put;
import org.apache.hadoop.hbase.client.RegionLocator;
import org.apache.hadoop.hbase.client.Result;
import org.apache.hadoop.hbase.client.ResultScanner;
import org.apache.hadoop.hbase.client.Row;
import org.apache.hadoop.hbase.client.RowMutations;
import org.apache.hadoop.hbase.client.Scan;
import org.apache.hadoop.hbase.client.Table;
import org.apache.hadoop.hbase.client.coprocessor.Batch;
import org.apache.hadoop.hbase.filter.CompareFilter;
import org.apache.hadoop.hbase.ipc.CoprocessorRpcChannel;
import org.apache.hadoop.hbase.protobuf.generated.AggregateProtos;
import org.apache.hadoop.hbase.util.Bytes;
import org.apache.hadoop.hbase.util.Pair;

public class AliHBaseUETable implements Table , RegionLocator {


  private TableName tableName;
  private String tableNameWithoutNamespace;
  private AliHBaseUEConnection connection;
  private Configuration conf;
  private int operationTimeout;
  private WideColumnService wideColumnService;
  private TableService tableService;
  private int defaultScannerCaching;

  public AliHBaseUETable(TableName tableName, AliHBaseUEConnection connection) throws IOException {
    this.tableName = tableName;
    this.connection = connection;
    this.conf = connection.getConfiguration();
    this.operationTimeout = connection.getOperationTimeout();
    this.wideColumnService = connection.getWideColumnService(tableName.getNamespaceAsString());
    this.tableService = connection.getTableServiceService(tableName.getNamespaceAsString());
    this.tableNameWithoutNamespace = tableName.getQualifierAsString();
    this.defaultScannerCaching = conf.getInt(HConstants.HBASE_CLIENT_SCANNER_CACHING,
        Constants.ALIHBASE_CLIENT_SCANNER_CACHING_DEFAULT);
    if (defaultScannerCaching > Constants.ALIHBASE_CLIENT_SCANNER_CACHING_DEFAULT) {
      this.defaultScannerCaching = Constants.ALIHBASE_CLIENT_SCANNER_CACHING_DEFAULT;
    }


  }

  @Override
  public HRegionLocation getRegionLocation(byte[] row) throws IOException {
    return getRegionLocation(row, false);
  }

  @Override
  public HRegionLocation getRegionLocation(byte[] row, boolean b) throws IOException {
    Pair<byte[], byte[]> startEndKey = getStartEndKey(row);
    return createFakeRegionLocation(startEndKey.getFirst(), startEndKey.getSecond());
  }

  private Pair<byte[], byte[]> getStartEndKey(byte[] row) throws IOException {
    byte[][] endKeys = getEndKeys();
    byte[] preStartKey = HConstants.EMPTY_START_ROW;
    for (byte[] endKey : endKeys) {
      if (Bytes.compareTo(row, endKey) < 0) {
        return new Pair<>(preStartKey, endKey);
      }
      preStartKey = endKey;
    }
    return new Pair<>(HConstants.EMPTY_START_ROW, HConstants.EMPTY_START_ROW);
  }

  private HRegionLocation createFakeRegionLocation(byte[] startKey, byte[] endKey) {
    HRegionInfo regionInfo = new HRegionInfo(tableName, startKey, endKey);
    return new HRegionLocation(regionInfo, ServerName.valueOf("localhost", 0, 0));
  }

  @Override
  public List<HRegionLocation> getAllRegionLocations() throws IOException {
    List<HRegionLocation> fakeRegionLocations = new ArrayList<>();
    byte[] preStartKey = HConstants.EMPTY_START_ROW;
    byte[][] endKeys = getEndKeys();
    for (int i = 0; i < endKeys.length; i++) {
      byte[] endKey = endKeys[i];
      fakeRegionLocations.add(createFakeRegionLocation(preStartKey, endKey));
      preStartKey = endKey;
    }
    return fakeRegionLocations;
  }

  @Override
  public byte[][] getStartKeys() throws IOException {
    return getStartEndKeys().getFirst();
  }

  @Override
  public byte[][] getEndKeys() throws IOException {
    return getStartEndKeys().getSecond();
  }

  @Override
  public Pair<byte[][], byte[][]> getStartEndKeys() throws IOException {
    com.alibaba.lindorm.client.core.utils.Pair<byte[][], byte[][]> pair = wideColumnService
        .getStartEndKeys(tableName.getQualifierAsString());
    return new Pair<byte[][], byte[][]>(pair.getFirst(), pair.getSecond());
  }

  @Override
  public TableName getName() {
    return tableName;
  }

  @Override
  public Configuration getConfiguration() {
    return conf;
  }

  @Override
  public HTableDescriptor getTableDescriptor() throws IOException {
    LindormTableDescriptor lindormTableDescriptor = wideColumnService
        .describeTable(tableName.getQualifierAsString());
    return ElementConvertor.toHbaseTableDescriptor(tableName.getNamespaceAsString(), lindormTableDescriptor);
  }

  @Override
  public boolean exists(Get get) throws IOException {
    WGet wGet = ElementConvertor.toLindormGet(get);
    return wideColumnService.exists(tableNameWithoutNamespace, wGet);
  }

  @Override
  public boolean[] existsAll(List<Get> gets) throws IOException {
    Result[] results = get(gets);
    boolean[] existResult = new boolean[results.length];
    for (int i = 0; i < results.length; i++) {
      existResult[i] = (results[i].getRow() != null);
    }
    return existResult;
  }


  @Override
  public <R> void batchCallback(List<? extends Row> actions, Object[] results,
      Batch.Callback<R> callback) throws IOException {
    throw new IOException("BatchCallback not supported, use put(List<Put> puts), "
        + "get(List<Get> gets) or delete(List<Delete> deletes) respectively");
  }

  @Override
  public Result get(Get get) throws IOException {
    WResult result = wideColumnService.get(tableNameWithoutNamespace,
        ElementConvertor.toLindormGet(get));
    return ElementConvertor.toHBaseResult(result);
  }

  @Override
  public Result[] get(List<Get> gets) throws IOException {
    WResult[] results = wideColumnService.batchGet(tableNameWithoutNamespace,
        ElementConvertor.toLindormGets(gets));
    return ElementConvertor.toHBaseResults(results);
  }

  /**
   * A scanner to perform scan from thrift server
   * getScannerResults is used in this scanner
   */
  private class Scanner implements ResultScanner {
    private final WScanner wScanner;
    protected Result lastResult = null;
    protected final Queue<Result> cache = new ArrayDeque<>();;


    public Scanner(Scan scan) throws IOException {
      if (scan.getCaching() <= 0) {
        scan.setCaching(defaultScannerCaching);
      } else if (scan.getCaching() == 1 && scan.isReversed()){
        // for reverse scan, we need to pass the last row to the next scanner
        // we need caching number bigger than 1
        scan.setCaching(scan.getCaching() + 1);
      }
      WScan wScan = ElementConvertor.toLindormScan(scan);
      this.wScanner = wideColumnService.getScanner(tableNameWithoutNamespace, wScan);
    }

    @Override
    public Iterator<Result> iterator() {
      return new Iterator<Result>() {
        private Iterator<WResult> it = wScanner.iterator();
        public boolean hasNext() {
          return it.hasNext();
        }

        public Result next() {
          WResult wResult = it.next();
          if (wResult == null) {
            return null;
          } else {
            return ElementConvertor.toHBaseResult(wResult);
          }
        }
      };
    }


    @Override
    public Result next() throws IOException {
      WResult r = wScanner.next();
      if (r == null) {
        return null;
      }
      return ElementConvertor.toHBaseResult(r);
    }

    @Override
    public void close() {
    }

    @Override
    public Result[] next(int nbRows) throws IOException {
      // Collect values to be returned here
      ArrayList<Result> resultSets = new ArrayList<Result>(nbRows);
      for(int i = 0; i < nbRows; i++) {
        Result next = next();
        if (next != null) {
          resultSets.add(next);
        } else {
          break;
        }
      }
      return resultSets.toArray(new Result[resultSets.size()]);
    }
  }

  @Override
  public ResultScanner getScanner(Scan scan) throws IOException {
    return new Scanner(scan);
  }

  @Override
  public ResultScanner getScanner(byte[] family) throws IOException {
    Scan scan = new Scan();
    scan.addFamily(family);
    return getScanner(scan);
  }

  @Override
  public ResultScanner getScanner(byte[] family, byte[] qualifier) throws IOException {
    Scan scan = new Scan();
    scan.addColumn(family, qualifier);
    return getScanner(scan);
  }

  @Override
  public void put(Put put) throws IOException {
    WPut wPut = ElementConvertor.toLindormPut(put);
    wideColumnService.put(tableNameWithoutNamespace, wPut);
  }

  @Override
  public void put(List<Put> puts) throws IOException {
    List<WPut> wPuts = ElementConvertor.toLindormPuts(puts);
    wideColumnService.batchPut(tableNameWithoutNamespace, wPuts);
  }

  @Override
  public void delete(Delete delete) throws IOException {
    WDelete wDelete = ElementConvertor.toLindormDelete(delete);
    wideColumnService.delete(tableNameWithoutNamespace, wDelete);
  }

  @Override
  public void delete(List<Delete> deletes) throws IOException {
    List<WDelete> wDeletes = ElementConvertor.toLindormDeleteList(deletes);
    wideColumnService.batchDelete(tableNameWithoutNamespace, wDeletes);
  }

  @Override
  public boolean checkAndMutate(byte[] row, byte[] family, byte[] qualifier,  CompareFilter.CompareOp op,
      byte[] value, RowMutations rowMutations) throws IOException {
    WRowMutations wRowMutations = ElementConvertor.toLindormRowMutation(rowMutations);
    WCompareFilter.CompareOp wOp = ElementConvertor.toLindormCompareOp(op);
    return wideColumnService
        .checkAndMutate(tableNameWithoutNamespace, row, family, qualifier, wOp, value,
            wRowMutations);
  }



  @Override
  public boolean checkAndPut(final byte [] row,
      final byte [] family, final byte [] qualifier, final byte [] value,
      final Put put)
      throws IOException {
    return checkAndPut(row, family, qualifier, CompareFilter.CompareOp.EQUAL, value, put);
  }

  @Override
  public boolean checkAndDelete(final byte [] row,
      final byte [] family, final byte [] qualifier, final byte [] value,
      final Delete delete) throws IOException {
    WDelete wDelete = ElementConvertor.toLindormDelete(delete);
    return wideColumnService.checkAndDelete(tableNameWithoutNamespace, row, family, qualifier, value, wDelete);
  }

  @Override
  public boolean checkAndPut(final byte [] row,
      final byte [] family, final byte [] qualifier,
      CompareFilter.CompareOp compareOp, final byte [] value,
      final Put put) throws IOException {
    WPut wPut = ElementConvertor.toLindormPut(put);
    WCompareFilter.CompareOp wOp = ElementConvertor.toLindormCompareOp(compareOp);
    return wideColumnService.checkAndPut(tableNameWithoutNamespace, row, family, qualifier, wOp, value, wPut);
  }

  @Override
  public boolean checkAndDelete(final byte [] row,
      final byte [] family, final byte [] qualifier,
      CompareFilter.CompareOp compareOp, final byte [] value, Delete delete) throws IOException {
    WDelete wDelete = ElementConvertor.toLindormDelete(delete);
    WCompareFilter.CompareOp wOp = ElementConvertor.toLindormCompareOp(compareOp);
    return wideColumnService.checkAndDelete(tableNameWithoutNamespace, row, family, qualifier, wOp, value, wDelete);
  }

  @Override
  public long incrementColumnValue(final byte [] row, final byte [] family,
      final byte [] qualifier, final long amount)
      throws IOException {
    return incrementColumnValue(row, family, qualifier, amount, Durability.SYNC_WAL);
  }

  @Override
  public long incrementColumnValue(final byte [] row, final byte [] family,
      final byte [] qualifier, final long amount, final Durability durability) throws IOException {
      Increment increment = new Increment(row);
      increment.addColumn(family, qualifier, amount);
      Result result =  increment(increment);
      return Long.valueOf(Bytes.toLong(result.getValue(family, qualifier)));
  }

  @Override
  public void mutateRow(RowMutations rm) throws IOException {
    for (Mutation mutation : rm.getMutations()) {
      if (mutation instanceof Put) {
        put((Put) mutation);
      } else if (mutation instanceof Delete) {
        delete((Delete) mutation);
      } else {
        throw new IOException("mutateRow not supported, use put(List<Put> puts), "
            + "get(List<Get> gets) or delete(List<Delete> deletes) respectively");
      }
    }
  }

  @Override
  public Result append(Append append) throws IOException {
    WAppend wAppend = ElementConvertor.toLindormAppend(append);
    WResult wResult = wideColumnService.append(tableNameWithoutNamespace, wAppend);
    return ElementConvertor.toHBaseResult(wResult);
  }

  @Override
  public Result increment(Increment increment) throws IOException {
    WIncrement wIncrement = ElementConvertor.toLindormIncrement(increment);
    WResult wResult = wideColumnService.increment(tableNameWithoutNamespace, wIncrement);
    return ElementConvertor.toHBaseResult(wResult);
  }

  @Override
  public void close() throws IOException {
  }

  @Override
  public CoprocessorRpcChannel coprocessorService(byte[] row) {
    throw new UnsupportedOperationException("coprocessorService not supported");
  }

  @Override
  public void batch(List<? extends Row> actions, Object[] results)
      throws IOException {
    List<Object> tactions = new ArrayList<Object>(actions.size());
    for (Row action : actions) {
      if (action instanceof Get) {
        tactions.add(ElementConvertor.toLindormGet((Get) action));
      } else if (action instanceof Put) {
        tactions.add(ElementConvertor.toLindormPut((Put) action));
      } else if (action instanceof Delete) {
        tactions.add(ElementConvertor.toLindormDelete((Delete) action));
      } else if (action instanceof Append) {
        tactions.add(ElementConvertor.toLindormAppend((Append) action));
      } else {
        throw new UnsupportedOperationException(
            "Unsupport type " + action.getClass().getName() + " in batch operation.");
      }
    }
    Object[] hbaseResults = null;
    BatchExceptions batchExceptions = new BatchExceptions();
    try {
      hbaseResults = batch(tableName.getQualifierAsString(), tactions);
    } finally {
      for (int i = 0; i < hbaseResults.length; i++) {
        if (results != null) {
          if (hbaseResults[i] == null) {
            results[i] = null;
          } else if (hbaseResults[i] instanceof WResult) {
            results[i] = ElementConvertor.toHBaseResult((WResult) hbaseResults[i]);
          } else if (hbaseResults[i] instanceof Result) {
            results[i] = hbaseResults[i];
          } else if (hbaseResults[i] instanceof Throwable) {
            results[i] = hbaseResults[i];
            batchExceptions.add((Throwable)hbaseResults[i], actions.get(i), null);
          } else {
            throw new IOException("Not unsupported result type " + hbaseResults[i]);
          }
        } else {
          if (hbaseResults[i] != null && hbaseResults[i] instanceof Throwable) {
            batchExceptions.add((Throwable)hbaseResults[i], actions.get(i), null);
          }
        }
      }
    }
    if (batchExceptions.hasErrors()) {
      throw batchExceptions.makeException();
    }
  }

  @Override
  public Object[] batch(List<? extends Row> actions) throws IOException {
    Object[] results = new Object[actions.size()];
    batch(actions, results);
    return results;
  }

  public Object[] batch(final String tableName, List<Object> actions) throws IOException {
    Object[] results = new Object[actions.size()];
    List<WGet> getActions = new ArrayList<>();
    List<Integer> getActionsIndex = new ArrayList<>();

    List<WPut> putActions = new ArrayList<>();
    List<Integer> putActionsIndex = new ArrayList<>();

    List<WDelete> delActions = new ArrayList<>();
    List<Integer> delActionsIndex = new ArrayList<>();

    List<WAppend> appendActions = new ArrayList<>();
    List<Integer> appendActionsIndex = new ArrayList<>();

    for (int i = 0; i < actions.size(); i++) {
      if (actions.get(i) instanceof WGet) {
        getActions.add((WGet) actions.get(i));
        getActionsIndex.add(i);
      } else if (actions.get(i) instanceof WPut) {
        putActions.add((WPut) actions.get(i));
        putActionsIndex.add(i);
      } else if (actions.get(i) instanceof WDelete) {
        delActions.add((WDelete) actions.get(i));
        delActionsIndex.add(i);
      } else if (actions.get(i) instanceof WAppend) {
        appendActions.add((WAppend) actions.get(i));
        appendActionsIndex.add(i);
      } else {
        throw new UnsupportedOperationException("Not supported action " + actions.get(i).getClass().getName());
      }
    }
    if (!getActions.isEmpty()) {
      try {
        WResult[] getResult = wideColumnService.batchGet(tableName, getActions);
        if (getResult.length != getActionsIndex.size()) {
          throw new IOException(
              "Batch get exception , expect " + getActionsIndex.size() + " results , current return " + ""
                  + getResult.length);
        }
        for (int i = 0; i < getActionsIndex.size(); i++) {
          results[getActionsIndex.get(i)] = getResult[i];
        }
      } catch (Throwable e) {
        for (Integer index : getActionsIndex) {
          results[index] = e;
        }
      }
    }
    if (!putActions.isEmpty()) {
      try {
        wideColumnService.batchPut(tableName, putActions);
        for (Integer index : putActionsIndex) {
          results[index] = new Result();
        }
      } catch (Throwable e) {
        for (Integer index : putActionsIndex) {
          results[index] = e;
        }
      }
    }
    if (!delActions.isEmpty()) {
      try {
        wideColumnService.batchDelete(tableName, delActions);
        for (Integer index : delActionsIndex) {
          results[index] = new Result();
        }
      } catch (Throwable e) {
        for (Integer index : delActionsIndex) {
          results[index] = e;
        }
      }
    }
    if (!appendActions.isEmpty()) {
      try {
        for (WAppend append : appendActions) {
          wideColumnService.append(tableName, append);
        }
        for (Integer index : appendActionsIndex) {
          results[index] = new Result();
        }
      } catch (Throwable e) {
        for (Integer index : appendActionsIndex) {
          results[index] = e;
        }
      }
    }
    return results;
  }

  @Override
  public <R> Object[] batchCallback(List<? extends Row> list, Batch.Callback<R> callback)
      throws IOException, InterruptedException {
    throw new UnsupportedOperationException("batchCallback not supported");
  }

  @Override
  public <T extends Service, R> Map<byte[], R> coprocessorService(Class<T> service, byte[] startKey, byte[] endKey,
      Batch.Call<T, R> call) throws ServiceException, Throwable {
    throw new UnsupportedOperationException("coprocessorService not supported");
  }

  @Override
  public <T extends Service, R> void coprocessorService(Class<T> service, byte[] startKey, byte[] endKey,
      Batch.Call<T, R> callable, Batch.Callback<R> callback) throws ServiceException, Throwable {
    if (service != AggregateProtos.AggregateService.class) {
      throw new UnsupportedOperationException("coprocessorService " + service + " not supported");
    }
    T instance = (T) new AliHBaseUEAggregateService(tableService, tableNameWithoutNamespace);
    R result = callable.call(instance);
    if (callback != null) {
      callback.update(null, null, result);
    }
  }

  @Override
  public long getWriteBufferSize() {
    throw new UnsupportedOperationException("getWriteBufferSize not supported");
  }

  @Override
  public void setWriteBufferSize(long l) throws IOException {
    throw new UnsupportedOperationException("setWriteBufferSize not supported");
  }

  @Override
  public <R extends Message> Map<byte[], R> batchCoprocessorService(
      Descriptors.MethodDescriptor methodDescriptor, Message message, byte[] startKey, byte[] endKey,
      R responsePrototype) throws ServiceException, Throwable {
    throw new UnsupportedOperationException("batchCoprocessorService not supported");
  }

  @Override
  public <R extends Message> void batchCoprocessorService(
      Descriptors.MethodDescriptor methodDescriptor, Message message, byte[] startKey, byte[] endKey,
      R responsePrototype, Batch.Callback<R> callback) throws ServiceException, Throwable {
    throw new UnsupportedOperationException("batchCoprocessorService not supported");
  }

  public void setOperationTimeout(int i) {
    throw new UnsupportedOperationException("setOperationTimeout not supported");
  }

  public int getOperationTimeout() {
    return connection.getOperationTimeout();
  }

  public int getRpcTimeout() {
    return connection.getOperationTimeout();
  }

  public void setRpcTimeout(int i) {
    throw new UnsupportedOperationException("setRpcTimeout not supported");
  }

  public int getReadRpcTimeout() {
    return connection.getOperationTimeout();
  }

  public void setReadRpcTimeout(int i) {
    throw new UnsupportedOperationException("setReadRpcTimeout not supported");
  }

  public int getWriteRpcTimeout() {
    return connection.getOperationTimeout();
  }

  public void setWriteRpcTimeout(int i) {
    throw new UnsupportedOperationException("setWriteRpcTimeout not supported");
  }

}
