/*
 * Copyright Alibaba Group Holding Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.alibaba.hbase.client;

import java.io.IOException;
import java.io.InterruptedIOException;
import java.lang.reflect.Constructor;
import java.net.UnknownHostException;
import java.util.Arrays;
import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.ExecutorService;

import javax.net.ssl.SSLException;

import com.alibaba.hbase.thrift2.generated.THBaseService;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.hbase.HConstants;
import org.apache.hadoop.hbase.TableName;
import org.apache.hadoop.hbase.client.Admin;
import org.apache.hadoop.hbase.client.BufferedMutator;
import org.apache.hadoop.hbase.client.BufferedMutatorParams;
import org.apache.hadoop.hbase.client.Connection;
import org.apache.hadoop.hbase.client.ConnectionUtils;
import org.apache.hadoop.hbase.client.RegionLocator;
import org.apache.hadoop.hbase.client.Table;
import org.apache.hadoop.hbase.security.User;
import org.apache.hadoop.hbase.util.Pair;
import org.apache.http.HttpRequest;
import org.apache.http.client.HttpClient;
import org.apache.http.client.config.RequestConfig;
import org.apache.http.client.utils.HttpClientUtils;
import org.apache.http.impl.client.DefaultHttpRequestRetryHandler;
import org.apache.http.impl.client.HttpClientBuilder;
import org.apache.http.protocol.HttpContext;
import org.apache.thrift.protocol.TBinaryProtocol;
import org.apache.thrift.protocol.TProtocol;
import org.apache.thrift.transport.THttpClient;
import org.apache.thrift.transport.TTransport;
import org.apache.thrift.transport.TTransportException;

public class AliHBaseThriftConnection implements Connection {
  public static final String ALIHBASE_THRIFT_CLIENT_BUIDLER_CLASS =
      "alihbase.thrift.client.builder.class";

  private Configuration conf;
  // For HTTP protocol
  private HttpClient httpClient;
  private boolean httpClientCreated = false;
  private boolean isClosed = false;

  private String host;

  private ThriftClientBuilder clientBuilder;

  private int operationTimeout;
  private int connectTimeout;

  // For HBase-1.x
  public AliHBaseThriftConnection(Configuration conf, boolean managed, ExecutorService pool,
      final User user) throws IOException {
    this(conf, pool, user);
  }

  public AliHBaseThriftConnection(Configuration conf, ExecutorService pool, final User user)
      throws IOException {
    this.conf = conf;
    this.host = conf.get(Constants.ALIHBASE_SERVER_NAME);
    if (host == null) {
      throw new RuntimeException(Constants.ALIHBASE_SERVER_NAME + " is not set");
    }
    host = host.trim();
    if (!host.startsWith("http://") && !host.startsWith("https://")) {
      throw new RuntimeException(
          Constants.ALIHBASE_SERVER_NAME + " should start with http:// or https://");
    }
    String hostOnly = host.split("//")[1];
    if (!hostOnly.contains(":")) {
      this.host = this.host + ":" + Constants.DEFAULT_THRIFT_PORT;
    }
    this.operationTimeout = conf.getInt(HConstants.HBASE_CLIENT_OPERATION_TIMEOUT,
        HConstants.DEFAULT_HBASE_CLIENT_OPERATION_TIMEOUT);
    this.connectTimeout = conf.getInt(Constants.SOCKET_TIMEOUT_CONNECT, Constants.DEFAULT_SOCKET_TIMEOUT_CONNECT);

    String className = conf.get(ALIHBASE_THRIFT_CLIENT_BUIDLER_CLASS,
        AliHBaseThriftClientBuilder.class.getName());
    try {
      Class<?> clazz = Class.forName(className);
      Constructor<?> constructor = clazz
          .getDeclaredConstructor(AliHBaseThriftConnection.class);
      constructor.setAccessible(true);
      clientBuilder = (ThriftClientBuilder) constructor.newInstance(this);
    }catch (Exception e) {
      throw new IOException(e);
    }
  }

  public synchronized void setHttpClient(HttpClient httpClient) {
    this.httpClient = httpClient;
  }

  @Override
  public Configuration getConfiguration() {
    return conf;
  }

  public String getHost() {
    return host;
  }

  public int getOperationTimeout() {
    return operationTimeout;
  }

  public int getConnectTimeout() {
    return connectTimeout;
  }

  public ThriftClientBuilder getClientBuilder() {
    return clientBuilder;
  }

  /**
   * Get a ThriftAdmin, ThriftAdmin is NOT thread safe
   * @return a ThriftAdmin
   * @throws IOException IOException
   */
  @Override
  public Admin getAdmin() throws IOException {
    Pair<THBaseService.Client, TTransport> client = clientBuilder.getClient();
    return new ThriftAdmin(client.getFirst(), client.getSecond(), conf);
  }

  /**
   * the default thrift http client builder. ONLY FOR TEST.
   * One can extend the ThriftClientBuilder to builder custom http client, implement
   * features like authentication or 'DoAs'(hbase-examples/thrift/HttpDoAsClient)
   *
   */
  public static class HTTPThriftClientBuilder extends ThriftClientBuilder {
    Map<String,String> customHeader = new HashMap<>();

    public HTTPThriftClientBuilder(AliHBaseThriftConnection connection) {
      super(connection);
    }

    public void addCostumHeader(String key, String value) {
      customHeader.put(key, value);
    }

    @Override
    public Pair<THBaseService.Client, TTransport> getClient() throws IOException {
      String url = connection.getHost();
      try {
        THttpClient httpClient = new THttpClient(url, connection.getHttpClient());
        for (Map.Entry<String, String> header : customHeader.entrySet()) {
          httpClient.setCustomHeader(header.getKey(), header.getValue());
        }
        httpClient.open();
        TProtocol prot = new TBinaryProtocol(httpClient);
        THBaseService.Client client = new THBaseService.Client(prot);
        return new Pair<>(client, httpClient);
      } catch (TTransportException e) {
        throw new IOException(e);
      }
    }
  }

  public static class DelayRetryHandler extends DefaultHttpRequestRetryHandler {
    private long pause;

    public DelayRetryHandler(int retryCount, long pause) {
      super(retryCount, true, Arrays.asList(
          InterruptedIOException.class,
          UnknownHostException.class,
          SSLException.class));
      this.pause = pause;
    }

    @Override
    public boolean retryRequest(IOException exception, int executionCount, HttpContext context) {
      // Don't sleep for retrying the first time
      if (executionCount > 1 && pause > 0) {
        try {
          long sleepTime = ConnectionUtils.getPauseTime(pause, executionCount - 1);
          Thread.sleep(sleepTime);
        } catch (InterruptedException ie) {
          //reset interrupt marker
          Thread.currentThread().interrupt();
        }
      }
      return super.retryRequest(exception, executionCount, context);
    }

    @Override
    protected boolean handleAsIdempotent(HttpRequest request) {
      return true;
    }
  }

  public synchronized HttpClient getHttpClient() {
    if (httpClient != null) {
      return httpClient;
    }
    int retry = conf.getInt(HConstants.HBASE_CLIENT_RETRIES_NUMBER,
        HConstants.DEFAULT_HBASE_CLIENT_RETRIES_NUMBER);
    long pause = conf.getLong(HConstants.HBASE_CLIENT_PAUSE, 5);
    HttpClientBuilder builder = HttpClientBuilder.create();
    RequestConfig.Builder requestBuilder = RequestConfig.custom();
    requestBuilder = requestBuilder.setConnectTimeout(getConnectTimeout());
    requestBuilder = requestBuilder.setSocketTimeout(getOperationTimeout());
    builder.setRetryHandler(new DelayRetryHandler(retry, pause));
    builder.setDefaultRequestConfig(requestBuilder.build());
    httpClient = builder.build();
    httpClientCreated = true;
    return httpClient;
  }

  @Override
  public synchronized void close() throws IOException {
    if (httpClient != null && httpClientCreated) {
      HttpClientUtils.closeQuietly(httpClient);
    }
    isClosed = true;
  }

  @Override
  public boolean isClosed() {
    return isClosed;
  }

  @Override
  public void abort(String why, Throwable e) {

  }

  @Override
  public boolean isAborted() {
    return false;
  }

  @Override
  public BufferedMutator getBufferedMutator(TableName tableName) throws IOException {
    throw new UnsupportedOperationException("batchCoprocessorService not supported in ThriftTable");
  }

  @Override
  public BufferedMutator getBufferedMutator(BufferedMutatorParams params) throws IOException {
    throw new UnsupportedOperationException("batchCoprocessorService not supported in ThriftTable");
  }

  @Override
  public RegionLocator getRegionLocator(TableName tableName) throws IOException {
    throw new UnsupportedOperationException("batchCoprocessorService not supported in ThriftTable");
  }

  // Available in HBase-2.2+
  public void clearRegionLocationCache() {
    throw new UnsupportedOperationException("clearRegionLocationCache not supported in ThriftTable");
  }

  // Only Available in HBase-1.x
  public Table getTable(TableName tableName) throws IOException {
    try {
      Pair<THBaseService.Client, TTransport> client = clientBuilder.getClient();
      return new ThriftTable(tableName, client.getFirst(), client.getSecond(), conf);
    } catch (IOException ioE) {
      throw new RuntimeException(ioE);
    }
  }

  // Only Available in HBase-1.x
  public Table getTable(TableName tableName, ExecutorService executorService) throws IOException {
    return getTable(tableName);
  }
}
